#include <algorithm>
#include <list>
#include <iostream>

class Observer {
public:
    virtual ~Observer() {}

    virtual void update() = 0;
};

class Subject {
public:
    virtual ~Subject() {}

    virtual void addObserver(Observer* o) = 0;

    virtual void removeObserver(Observer* o) = 0;

    virtual void notifyObservers() = 0;
};

struct DiskFile {
    DiskFile(std::string n = "", float fs = 0.) : name(n), fileSize(fs) {}

    bool operator==(const DiskFile &right) const {
        if ((name == right.name) && (fileSize == right.fileSize))
            return true;
        else
            return false;
    }

    std::string name;
    float fileSize; // size in MB
};

class Filesystem : public Subject {
public:
    explicit Filesystem(float size) : freeSize(size), diskSize(size) {};

    int getNumFiles() const {
        return files.size();
    }

    float getFreeSpace() const {
        return freeSize;
    }

    float getOccupiedSpace() const {
        return diskSize - freeSize;
    }

    void create(DiskFile newFile) {
        if (freeSize > newFile.fileSize) {
            files.push_back(newFile);
            freeSize -= newFile.fileSize;
            notifyObservers();
        }
    }

    void remove(DiskFile oldFile) {
        auto it = find(files.begin(), files.end(), oldFile);
        if (it != files.end()) {
            files.remove(oldFile);
            freeSize += oldFile.fileSize;
            notifyObservers();
        }
    }

    bool rename(DiskFile oldFile, std::string newName) {
        std::list<DiskFile>::iterator it;
        it = find(files.begin(), files.end(), oldFile);
        if (it != files.end()) {
            it->name = newName;
            return true;
        }
        return false;
    }

    void printAll() {
        for (auto it : files)
            std::cout << it.name << std::endl;
    }

    virtual void addObserver(Observer* o) override {
        observers.push_back(o);
    }

    virtual void removeObserver(Observer* o) override {
        observers.remove(o);
    }

    virtual void notifyObservers() override {
        for (auto observer : observers)
            observer->update();
    }

private:
    std::list<DiskFile> files;
    float diskSize; // disk size in MB
    float freeSize; // free space in MB

    std::list<Observer*> observers;
};

class LowDiskSpaceWarning : public Observer {
public:
    LowDiskSpaceWarning(Filesystem* subject) : subject(subject) {
        subject->addObserver(this);
        minFreeSize = 10.;
        first = true;
    }

    virtual ~LowDiskSpaceWarning() {
        subject->removeObserver(this);
    }

    virtual void update() override {
        if (subject->getFreeSpace() < minFreeSize) {
            std::cout << "Free disk space low (" << subject->getFreeSpace() << "< " << minFreeSize << ")" << std::endl;
            minFreeSize = subject->getFreeSpace();
        }
        if (subject->getFreeSpace() >= 10)
            minFreeSize = 10;
    }

private:
    Filesystem* subject;
    float minFreeSize;
    bool first;
};

class FilesystemInfo : public Observer {
public:
    FilesystemInfo(Filesystem* subject) : subject(subject) {
        subject->addObserver(this);
    }

    virtual ~FilesystemInfo() {
        subject->removeObserver(this);
    }

    void update() override {
        std::cout << "Files: " << subject->getNumFiles() << " - Bytes: " << subject->getOccupiedSpace() << " - Free: "
                  << subject->getFreeSpace() << std::endl;
    }

private:
    Filesystem* subject;
};

int main(int argc, char* argv[]) {
    DiskFile df1("foo.txt", 1);
    DiskFile df2("bar.log", 20);
    Filesystem fs(50);
    LowDiskSpaceWarning ldsw(&fs);
    FilesystemInfo fi(&fs);
    fs.create(df1);
    std::cout << fs.getNumFiles() << std::endl;
    std::cout << fs.getFreeSpace() << std::endl;
    fs.create(df2);
    std::cout << fs.getNumFiles() << std::endl;
    std::cout << fs.getFreeSpace() << std::endl;
    DiskFile df3("fizz.jpg", 20);
    fs.create(df3);
    fs.rename(df3, "buzz.jpg");
    DiskFile df4("fizz.jpg", 4);
    fs.create(df4);
    fs.remove(df4);
    fs.remove(df3);
    fs.create(df3);
    fs.create(df4);
    fs.printAll();
}
